var FormValidator = (function (form) {
    var root = this;
    root.form = form;
    root.listeners = {};

    root.addEventListener = function (event, callback) {
        if (!root.listeners[event]) {
            root.listeners[event] = []
        }

        root.listeners[event].push(callback)
    }

    root.dispatchEvent = function (event, data) {
        if (!root.listeners[event]) {
            return;
        }

        for (var i = 0; i < root.listeners[event].length; i++) {
            root.listeners[event][i](data)
        }
    }   

    root.getFormData = function () {
        var formData = {};
        var elements = root.form.querySelectorAll('input, select, textarea')
        for (var i = 0; i < elements.length; i++) {
            var element = elements[i];

            if (element.type == "checkbox") {
                formData[element.name] = element.checked
                continue;
            }

            formData[element.name] = element.value
        }

        return formData;
    }

    root.setFormData = function (data) {
        for (var key in data) {
            var element = root.form.querySelector('[name="' + key + '"]');
            if (!element) {
                continue;
            }

            if (element.type == "checkbox") {
                element.checked = data[key]
                continue;
            }

            element.value = data[key]
        }
    }

    root.DefaultValidations = {
        "required": function () {
            return {
                validate: function (value) {
                    return value.replace(/[^\x00-\x7F]/g, "").trim() != ""
                },
                message: "{field} is required"
            };
        },
        "matrixid": function () {
            return {
                validate: function (value) {
                    return /^[a-z0-9\._=\-:@#\/-]+$/.test(value)
                },
                message: "{field} is not a valid Matrix ID."
            }
        },
        "email": function () {
            return {
                validate: function (value) {
                    return value.trim().match(/^[a-zA-Z0-9._-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,6}$/)
                },
                message: "{field} is not a valid email address"
            }
        },
        "password": function () {
            return {
                validate: function (value, values, validator) {
                    validator.validateField("confirmPassword")
                    return value.trim().length >= 6
                },
                message: "{field} must be at least 6 characters long"
            }
        },
        "confirmPassword": function () {
            return {
                validate: function (value, allValues) {
                    return value.trim() == allValues.password.trim()
                },
                message: "{field} does not match password"
            }
        },
        "pin": function () {
            return {
                validate: function (value) {
                    value = value.trim();
                    if (value.length != 8) {
                        return false;
                    }

                    var firstCharIsNumber = value[0].match(/[0-9]/)
                    if (!firstCharIsNumber) {
                        return false;
                    }

                    var containsAtLeastALetter = value.match(/[a-zA-Z]/)
                    if (!containsAtLeastALetter) {
                        return false;
                    }

                    return true;
                },
                message: "{field} must be valid."
            }
        },
        "maxLength": function (length) {
            return {
                validate: function (value) {
                    return value.trim().length <= length
                },
                message: "{field} must be at most " + length + " characters long"
            }
        },
        "minLength": function (length) {
            return {
                validate: function (value) {
                    return value.trim().length >= length
                },
                message: "{field} must be at least " + length + " characters long"
            }
        },
        "exactLength": function (length) {
            return {
                validate: function (value) {
                    return value.trim().length == length
                },
                message: "{field} must be exactly " + length + " characters long"
            }
        },
        "numeric": function () {
            return {
                validate: function (value) {
                    return value.trim().match(/^[0-9]+$/)
                },
                message: "{field} must be a number"
            }
        },
        "decimal": function () {
            return {
                validate: function (value) {
                    return value.trim().match(/^[0-9]+(\.[0-9]+)?$/)
                },
                message: "{field} must be a decimal number"
            }
        },
        "positive": function () {
            return {
                validate: function (value) {
                    return parseFloat(value) > 0
                },
                message: "{field} must be a positive number"
            }
        },
        "negative": function () {
            return {
                validate: function (value) {
                    return parseFloat(value) < 0
                },
                message: "{field} must be a negative number"
            }
        },
        "alphaNumeric": function () {
            return {
                validate: function (value) {
                    return value.trim().match(/^[a-zA-Z0-9]+$/)
                },
                message: "{field} must be alphanumeric"
            }
        },
        "alpha": function () {
            return {
                validate: function (value) {
                    return value.trim().match(/^[a-zA-Z]+$/)
                },
                message: "{field} must be alphabetic"
            }
        },
        "url": function () {
            return {
                validate: function (value) {
                    return value.trim().match(/^(http|https):\/\/[a-zA-Z0-9.-]+\.[a-zA-Z]{2,6}$/)
                },
                message: "{field} must be a valid URL"
            }
        },
        "date": function () {
            return {
                validate: function (value) {
                    return value.trim().match(/^[0-9]{4}-[0-9]{2}-[0-9]{2}$/)
                },
                message: "{field} must be a valid date"
            }
        },
        "time": function () {
            return {
                validate: function (value) {
                    return value.trim().match(/^[0-9]{2}:[0-9]{2}$/)
                },
                message: "{field} must be a valid time"
            }
        },
        "dateTime": function () {
            return {
                validate: function (value) {
                    return value.trim().match(/^[0-9]{4}-[0-9]{2}-[0-9]{2} [0-9]{2}:[0-9]{2}$/)
                },
                message: "{field} must be a valid date and time"
            }
        },
        "pattern": function (pattern) {
            return {
                validate: function (value) {
                    return value.trim().match(pattern)
                },
                message: "{field} does not match the required pattern"
            }
        }
    }

    var validations = [];

    root.addValidation = function (field, validation) {
        var fieldValidations = validations.find(function (validation) {
            return validation.field == field
        })

        if (!fieldValidations) {
            fieldValidations = {
                field: field,
                validations: []
            }
            validations.push(fieldValidations)
        }


        if (typeof validation == "string") {
            var args = [];
            if (validation.indexOf(":") > -1) {
                var composition = validation.split(":");
                args = composition[1].split(",");
                validation = composition[0]
            }
            validation = root.DefaultValidations[validation].apply(null, args)
        }

        fieldValidations.validations.push(validation)
        return root;
    }

    root.validateField = function (field) {
        var fieldValidations = validations.find(function (validation) {
            return validation.field == field
        })

        if (!fieldValidations) {
            return true;
        }

        var isValid = true;
        var formData = root.getFormData();
        for (var i = 0; i < fieldValidations.validations.length; i++) {
            var validation = fieldValidations.validations[i];
            var element = root.form.querySelector('[name="' + fieldValidations.field + '"]');
            var validationElement = element.parentElement.querySelector('.validation-message');
            var fieldLabel = element.parentElement.querySelector('.form-label');
            if (!validation.validate(formData[field], formData, root)) {
                isValid = false;
                if (validationElement) {
                    var message = validation.message;
                    if (fieldLabel) {
                        message = message.replace("{field}", fieldLabel.innerText.trim())
                    }
                    validationElement.innerHTML = message;
                }
                break;
            } else {
                if (validationElement) {
                    validationElement.innerHTML = ""
                }
            }
        }

        return isValid;
    }

    root.validate = function () {
        var isValid = true;
        for (var i = 0; i < validations.length; i++) {
            var fieldValidations = validations[i];
            if (!root.validateField(fieldValidations.field)) {
                isValid = false;
            }
        }

        return isValid;
    }

    root.validate();

    root.submit = function () {
        if (root.validate()) {
            root.dispatchEvent('submit', root.getFormData())
        }
    }

    root.form.addEventListener('input', function (e) {
        root.validateField(e.target.name)
    })

    root.form.addEventListener('submit', function (e) {
        e.preventDefault();
        root.submit();
        return false;
    })
    
    return root;
})